import math

# Define the HDGL Machine State
class HDGLMachine:
    def __init__(self):
        # Upper Field: Expansion Primitives
        self.upper_field = {
            "prism_state": 105.0,  # Composite R+G+B DNA
            "recursion_mode": 99.9999999999,  # Recursion Mode Switch
            "positive_infinite": 9.9999999999,  # Positive infinite recursion slot
            "P3": 4.2360679775,  # Fibonacci 5 DNA Braid
            "pi": 3.1415926535,  # Conditional and Logical Operator
            "phi_power_phi": 2.6180339887,  # Addition Operator
            "phi": 1.6180339887  # Upper boundary and primary primitive
        }

        # Analog Dimensionality Primitives (Active State)
        self.analog_dims = {
            "D8": 8.0,  # Full hybrid lattice
            "D7": 7.0,  # Strand A+B superposition
            "D6": 6.0,  # Strand B, Wave -
            "D5": 5.0,  # Strand B, Wave 0
            "D4": 4.0,  # Strand B, Wave +
            "D3": 3.0,  # Strand A, Wave -
            "D2": 2.0,  # Strand A, Wave 0
            "D1": 1.0,  # Strand A, Wave +
            "dim_switch": 1.0,  # Dimensionality Switch (1 = 2D double-helix)
            "P4": 6.8541019662,  # Fibonacci 8 DNA Braid
            "P5": 11.0901699437,  # Fibonacci 13 DNA Braid
            "P6": 17.94427191,  # Fibonacci 21 DNA Braid
            "P7": 29.0344465435  # Fibonacci 34 DNA Braid
        }

        # The Void
        self.void = 0.0

        # Lower Field: Contraction Primitives
        self.lower_field = {
            "negative_infinite": 0.0000000001,  # Negative infinite recursion slot
            "inv_P7": 0.0344465435,  # 1/P7
            "inv_P6": 0.05572809,  # 1/P6
            "inv_P5": 0.0901699437,  # 1/P5
            "inv_P4": 0.1458980338,  # 1/P4
            "inv_P3": 0.2360679775,  # 1/P3
            "inv_phi_power_phi": 0.3819660113,  # 1/φ^φ
            "inv_phi": 0.6180339887  # Lower boundary
        }

        # Current state
        self.current_state = self.void
        self.dimension = self.analog_dims["dim_switch"]  # 2D double-helix by default
        self.recursion_active = True

    def toggle_recursion(self):
        """Toggle recursion mode based on recursion_mode value."""
        self.recursion_active = self.recursion_active
        print(f"Recursion mode: {'ON' if self.recursion_active else 'OFF'}")

    def compute_harmonic_state(self, t):
        """Compute the harmonic state as a superposition of primitives at time t."""
        # Combine upper and lower fields with a harmonic oscillation
        state = self.void
        for key, value in self.upper_field.items():
            state += value * math.sin(t * self.upper_field["phi"])
        for key, value in self.lower_field.items():
            state += value * math.cos(t * self.lower_field["inv_phi"])
        for key, value in self.analog_dims.items():
            state += value * math.sin(t * self.upper_field["phi_power_phi"])

        # Apply dimensionality switch
        if self.dimension == 1.0:  # 2D double-helix
            state *= math.sin(t * self.upper_field["pi"])
        else:  # 1D
            state *= math.cos(t * self.upper_field["pi"])

        return state

    def step(self, t):
        """Advance the machine state for time step t."""
        if self.recursion_active:
            # Recursive state update using Fibonacci braids
            self.current_state = self.compute_harmonic_state(t)
            self.current_state *= self.analog_dims["P7"] / self.lower_field["inv_P7"]
        else:
            # Non-recursive update
            self.current_state = self.compute_harmonic_state(t)
        print(f"Time {t:.2f}: State = {self.current_state:.6f}")

    def run_simulation(self, steps=10, dt=0.1):
        """Run a simulation of the harmonic stream for a given number of steps."""
        print("Starting HDGL Machine Simulation")
        for i in range(steps):
            self.step(i * dt)
        print("Simulation complete")

# Bootstrap and run the machine
if __name__ == "__main__":
    machine = HDGLMachine()
    print("HDGL Machine Initialized")
    print(f"Initial State: {machine.current_state}")
    print(f"Dimensionality: {'2D double-helix' if machine.dimension == 1.0 else '1D'}")
    
    # Toggle recursion mode to demonstrate
    machine.toggle_recursion()
    
    # Run a short simulation
    machine.run_simulation(steps=10000, dt=0.001)